# ------------------------------------------------------------------------
# TOOLS FUNCTIONS
# description: Tools functions: string treatment, etc ..
# author : Juliette Fabre / OSU OREME
# creation : 11/2012
# last update : 04/2017
# ------------------------------------------------------------------------

# Remove multiple spaces and spaces at the beginning and end of a vector of strings
#
# Arguments:
# - x: vector of strings
#
# Values:
# - vector of strings

trim <- function(x) 
{
  gsub("[[:space:]]+", " ", gsub( "^[[:space:]]+|[[:space:]]+$", "", x))
}	

# Return indices of lines that contain a string s in a vector of strings
#
# Arguments:
# - x: vector of strings
#
# Values:
# - integer

get_line <- function(x, s)
{
	return(which(regexpr(s, x) != -1))
}

# Return the content of lines that contain a string in a vector of strings
#
# Arguments:
# - x: vector of strings
# - s: string to search
#
# Values:
# - vector of strings

get_line_c <- function(x, s)
{
	return(x[which(regexpr(s, x) != -1)])
}

# Return the remaining content after replacement of a string in all lines that contain it in a vector of strings
#
# Arguments:
# - x: vector of strings
# - s: string to search
#
# Values:
# - vector of strings

get_line_r <- function(x, s)
{
	return(trim(gsub(s, "", get_line_c(x, s))))
}

# Tests if values of a vector look like a year
#
# Arguments:
# - x: vector
#
# Values:
# - boolean

is_year <- function(x)
{
	options(warn = -1)
	return(is.na(x) | (regexpr("^[0-9]{4}$", x) != -1 & as.numeric(x) <= 3000))
}

# Return a vector of file names without their extension
#
# Arguments:
# - x: vector of file names
#
# Values:
# - vector of strings

get_file_without_ext <- function(x)
{
	return(gsub("[.]\\w+$", "", x))
}

# Return a vector of file name extensions
#
# Arguments:
# - x: vector of file names
#
# Values:
# - vector of strings

get_file_ext <- function(x)
{
  return(gsub("^[-\\(\\)°[:alnum:]_[:space:]]*[.]", "", x))
}

# Return a vector of file names without the complete path
#
# Arguments:
# - x: vector of file names
#
# Values:
# - vector of strings

get_file_without_path <- function(x)
{ 
  get_file <- function(s)
  {
    return(s[length(s)])
  }
  sapply(strsplit(x, '/') , get_file)
}

# Removes final ".0" from a string vector
#
# Arguments:
# - x: vector of strings
#
# Values:
# - vector of strings

remove_final_zero <- function(x)
{
  return(gsub("\\.0$", "", x))
}

# Capitalize a vector of strings (toupper every first letter of a word ) = capwords function from ?toupper
#
# Arguments:
# - x: vector of strings
# - strict: boolean, default false. If true, tolowers all other letters
#
# Values:
# - vector of strings

capitalize <- function(s, strict = FALSE) 
{
  cap <- function(s)
  {
    paste(toupper(substring(s, 1, 1)), {s <- substring(s, 2); if(strict) tolower(s) else s}, sep = "", collapse = " " )
  }
  sapply(strsplit(s, split = " "), cap, USE.NAMES = !is.null(names(s)))
}

# Toupper the first letter in a vector of strings
#
# Arguments:
# - x: vector of strings
# - strict: boolean, default false If true, tolowers all other letters
#
# Values:
# - vector of strings

capitalize_first <- function(s, strict = FALSE) 
{
  cap <- function(s)
  {
    if(!is.na(s)) return(paste(toupper(substring(s, 1, 1)), {s <- substring(s, 2); if(strict) tolower(s) else s}, sep = "", collapse = " " )) else return(s)
  }
  sapply(s, cap, USE.NAMES = !is.null(names(s)))
}

# Toupper the first letter of each word in a vector of strings
#
# Arguments:
# - x: vector of strings
# - strict: boolean, default false If true, tolowers all other letters
#
# Values:
# - vector of strings

capitalize_first_word <- function(s, strict = FALSE)
{
  cap <- function(s)
  {
    if(!is.na(s)) {
      x <- strsplit(s, " ")[[1]]
      return(paste0(toupper(substring(x, 1, 1)), {x <- substring(x, 2); if(strict) tolower(x) else x}, collapse = " ")) 
    } else return(s)
  }
  
  sapply(s, cap, USE.NAMES = !is.null(names(s)))
}

# Add quotes to a character vector, just before and after
#
# Arguments:
# - x: vector of strings
#
# Values:
# - vector of strings

add_quotes <- function(x)
{
  add_quote <- function(s)
  {
    paste('\"', s, '\"', sep = "")
  }
  sapply(x, add_quote, USE.NAMES = !is.null(names(x)))
}

# Add a given number of month to a date
#
# Arguments:
# - date: date
# - n: number of months to add (can be negative)
#
# Values:
# - date
add_months <- function(date,n) seq(date, by = paste (n, "months"), length = 2)[2]
